﻿using Microsoft.Crm.Sdk.Messages;
using Microsoft.Xrm.Sdk;
using Microsoft.Xrm.Sdk.Client;
using System;
using System.Linq;
using VA.TMP.DataModel;
using VA.TMP.Integration.VIMT.HealthShare.StateObject;
using VA.TMP.Integration.VIMT.Messages.HealthShare;
using VA.TMP.OptionSets;
using VRM.Integration.Servicebus.Core;

namespace VA.TMP.Integration.VIMT.HealthShare.PipelineSteps.MakeCancel
{
    /// <summary>
    /// Create and Save CRM entities.
    /// </summary>
    public class CreateAndSaveEntitiesStep : FilterBase<MakeCancelStateObject>
    {
        /// <summary>
        /// Execute the step.
        /// </summary>
        /// <param name="state">State object.</param>
        public override void Execute(MakeCancelStateObject state)
        {
            try
            {
                if (state.Appointment == null)
                {
                    Logger.Instance.Debug("MakeCancelStateObject.Appointment is null. Cannot process VA.TMP.Integration.VIMT.HealthShare.PipelineSteps.MakeCancel.CreateAndSaveEntities.cs");
                }
                else
                {
                    var appointment = state.Appointment;
                    var orgContext = new OrganizationServiceContext(state.OrganizationServiceProxy);

                    if (appointment.ScheduledDurationMinutes != null && appointment.ScheduledStart != null)
                            appointment.ScheduledEnd = appointment.ScheduledStart.Value.AddMinutes(
                                    (double) appointment.ScheduledDurationMinutes);

                    appointment.Subject = $"{appointment.cvt_ConsultName} @ {appointment.cvt_FacilityText}";

                    if (appointment == null)
                    {
                        const string err = "Appointment is null. Cannot process Make/Cancel Appointment at step VA.TMP.Integration.VIMT.HealthShare.PipelineSteps.MakeCancel.CreateandSaveEntities.cs";
                        Logger.Instance.Error(err);
                    }
                    else
                    {
                        var status = state.RequestMessage?.VisitStatus;
                        if (!string.IsNullOrWhiteSpace(status) && status.ToUpperInvariant() != "SCHEDULED")
                        {
                            //we have to search for the *existing appointment to cancel 
                            using (var service = new Xrm(state.OrganizationServiceProxy))
                            {
                                var existingAppointment = service.AppointmentSet.FirstOrDefault(
                                     i => i.cvt_ClinicIEN == appointment.cvt_ClinicIEN &&
                                          i.cvt_PatientICN == appointment.cvt_PatientICN &&
                                          i.ScheduledStart == ((DateTime)appointment.ScheduledStart).ToUniversalTime() &&
                                          i.StateCode != AppointmentState.Canceled
                                    );

                                if (existingAppointment != null) //We found the appointment, now we can cancel it.
                                {
                                    var req = new SetStateRequest
                                    {
                                        EntityMoniker = new EntityReference("appointment", existingAppointment.Id),
                                        State = new OptionSetValue((int)AppointmentState.Canceled),
                                        Status = new OptionSetValue((int)appointment_statuscode.Canceled)
                                    };

                                    service.Execute(req);
                                    service.UpdateObject(existingAppointment);

                                    CreateIntegrationResult(existingAppointment.Id, state, orgContext);
                                }
                                else
                                {
                                    Logger.Instance.Info($"Error: Could not find an appointment with Clinic IEN: {appointment.cvt_ClinicIEN}, PatientICN: {appointment.cvt_PatientICN}, ScheduledStart: {appointment.ScheduledStart}");
                                    throw new Exception($"Error: Could not find an appointment with Clinic IEN: {state.RequestMessage?.ClinicIen}, PatientICN: {state.RequestMessage?.PatientIcn}, ScheduledStart: {state.RequestMessage?.StartTime}");
                                }
                            }
                        }
                        else //we can schedule one
                        {
                            orgContext.AddObject(appointment);
                            orgContext.SaveChanges();


                            using (var service = new Xrm(state.OrganizationServiceProxy))
                            {
                                var cAppt = service.AppointmentSet.FirstOrDefault(i => i.ActivityId == appointment.ActivityId);

                                if (cAppt != null)
                                {
                                    var req = new SetStateRequest
                                    {
                                        EntityMoniker = new EntityReference("appointment", cAppt.Id),
                                        State = new OptionSetValue((int)AppointmentState.Scheduled),
                                        Status = new OptionSetValue((int)appointment_statuscode.Busy)//osvStatus
                                    };
                                    service.Execute(req);
                                }

                                service.UpdateObject(cAppt);

                            }
                            orgContext.Attach(appointment);

                            CreateIntegrationResult(appointment.ActivityId, state, orgContext);
                        }

                    }

                }

            }
            catch (Exception ex)
            {
                throw new Exception(string.Format("HealthShare Make Cancel CreateAndSaveEntitiesStep Pipeline Error: {0}", ex.Message), ex.InnerException);
            }
        }

        /// <summary>
        /// Create the Integration Result Record
        /// </summary>
        /// <param name="apptId">Appointment Record Id</param>
        /// <param name="state">MakeCancelStateObject</param>
        /// <param name="orgContext">Org Context.</param>
        /// <returns>the Integration Result Object created </returns>
        private static void CreateIntegrationResult(Guid? apptId, MakeCancelStateObject state, OrganizationServiceContext orgContext)
        {
            try
            {
                var integrationResult = new mcs_integrationresult
                {
                    mcs_name = $"HealthShare MakeCancel Appointment - {state.Appointment?.cvt_ConsultName}",
                    mcs_integrationrequest = state.SerializedRequestMessage,
                    mcs_VimtRequestMessageType = typeof(TmpHealthShareMakeAndCancelAppointmentRequestMessage).FullName,
                    mcs_VimtResponseMessageType = typeof(TmpHealthShareMakeAndCancelAppointmentResponseMessage).FullName,
                    mcs_VimtMessageRegistryName = MessageRegistry.TmpHealthShareMakeAndCancelAppointmentRequestMessage,
                    mcs_status=new OptionSetValue(803750002)
                };


                if (apptId != null && apptId != Guid.Empty)
                {
                    integrationResult.mcs_appointmentid = new EntityReference(Appointment.EntityLogicalName, apptId.Value);
                }

                orgContext.AddObject(integrationResult);
                orgContext.SaveChanges();

                var vistaIntegrationResult = CreateVistaIntegrationResult(apptId, integrationResult, state);

                orgContext.AddObject(vistaIntegrationResult);
                orgContext.SaveChanges();
            }
            catch (Exception ex)
            {
                throw new Exception($"Error creating Integration Result during HealthShare Make/Cancel Appointment. Details: {ex.Message}", ex.InnerException);
            }
        }

        /// <summary>
        /// Create the Vista Integration Result Record
        /// </summary>
        /// <param name="apptId">Appointment Record Id</param>
        /// /// <param name="ir">Associated Integration Record</param>
        /// <param name="state">MakeCancelStateObject</param>
        /// <returns>the Integration Result Object created </returns>
        private static cvt_vistaintegrationresult CreateVistaIntegrationResult(Guid? apptId, mcs_integrationresult ir, MakeCancelStateObject state)
        {
            using (var service = new Xrm(state.OrganizationServiceProxy))
            {
                var patient = (service.mcs_personidentifiersSet.FirstOrDefault(i => i.mcs_assigningauthority == "USVHA" &&
                                                                                  i.mcs_identifier == state.Appointment.cvt_PatientICN));

                var vistaIntegrationResult = new cvt_vistaintegrationresult();
                var appt = state.Appointment;
                try
                {
                    vistaIntegrationResult.cvt_ClinicIEN = appt.cvt_ClinicIEN;
                    vistaIntegrationResult.cvt_ClinicName = appt.cvt_ClinicName;
                    vistaIntegrationResult.cvt_DateTime = String.Format("{0:yyyymmddhh}", DateTime.Now);
                    if(appt.cvt_Facility != null)
                        vistaIntegrationResult.cvt_FacilityCode = appt.cvt_Facility.Name;
                    vistaIntegrationResult.cvt_FacilityName = appt.cvt_ClinicName;
                    vistaIntegrationResult.cvt_PatientName = appt.cvt_BookedPatients;
                    vistaIntegrationResult.cvt_PersonId = appt.cvt_PatientICN;
                    vistaIntegrationResult.cvt_Reason = appt.cvt_CancelReason;
                    vistaIntegrationResult.cvt_VistAStatus = appt.cvt_VisitStatus;
                    vistaIntegrationResult.cvt_ParentResult = new EntityReference { Id = ir.Id, LogicalName = mcs_integrationresult.EntityLogicalName };
                    vistaIntegrationResult.cvt_name = $"{appt.cvt_ConsultName} - at {appt.cvt_ClinicName} ({appt.cvt_FacilityText})";
                    vistaIntegrationResult.cvt_VistaReasonCode = appt.cvt_CancelReason == "UNKNOWN" ? "" : appt.cvt_CancelReason;
                    vistaIntegrationResult.cvt_VistaStatusCode = appt.cvt_CancelCode == "UNKNOWN" ? "" : appt.cvt_CancelCode;
                    if(patient != null && patient.mcs_patient != null)
                        vistaIntegrationResult.cvt_Veteran = patient.mcs_patient;
                    if (apptId != null)
                    {
                        vistaIntegrationResult.cvt_Appointment = new EntityReference("appointment", apptId.Value);
                    }
                }
                catch (Exception ex)
                {
                    throw new Exception($"Error creating Vista Integration Result during HealthShare Make/Cancel Appointment. Details: {ex.Message}", ex.InnerException);
                }

                return vistaIntegrationResult;

            }
        }
    }
}